"use client"

import { useState } from "react"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Textarea } from "@/components/ui/textarea"
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select"
import { CheckCircle2 } from "lucide-react"

export function ClientContactForm() {
  const [formState, setFormState] = useState({
    name: "",
    email: "",
    phone: "",
    service: "",
    message: "",
  })

  const [errors, setErrors] = useState({})
  const [isSubmitting, setIsSubmitting] = useState(false)
  const [isSubmitted, setIsSubmitted] = useState(false)

  const handleChange = (e) => {
    const { name, value } = e.target
    setFormState((prev) => ({
      ...prev,
      [name]: value,
    }))
  }

  const handleSelectChange = (value) => {
    setFormState((prev) => ({
      ...prev,
      service: value,
    }))
  }

  const validateForm = () => {
    const newErrors = {}

    if (!formState.name.trim()) {
      newErrors.name = "Nome é obrigatório"
    }

    if (!formState.email.trim()) {
      newErrors.email = "Email é obrigatório"
    } else if (!/\S+@\S+\.\S+/.test(formState.email)) {
      newErrors.email = "Email inválido"
    }

    if (!formState.phone.trim()) {
      newErrors.phone = "Telefone é obrigatório"
    }

    if (!formState.service) {
      newErrors.service = "Selecione um serviço"
    }

    setErrors(newErrors)
    return Object.keys(newErrors).length === 0
  }

  const handleSubmit = (e) => {
    e.preventDefault()

    if (validateForm()) {
      setIsSubmitting(true)

      // Simulate API call
      setTimeout(() => {
        setIsSubmitting(false)
        setIsSubmitted(true)

        // Reset form after successful submission
        setFormState({
          name: "",
          email: "",
          phone: "",
          service: "",
          message: "",
        })
      }, 1500)
    }
  }

  if (isSubmitted) {
    return (
      <div className="flex flex-col items-center justify-center py-8 text-center">
        <CheckCircle2 className="h-16 w-16 text-green-500 mb-4" />
        <h3 className="text-xl font-semibold mb-2">Mensagem enviada com sucesso!</h3>
        <p className="text-gray-600 mb-6">Obrigado pelo seu contato. Nossa equipe entrará em contato em breve.</p>
        <Button onClick={() => setIsSubmitted(false)} className="bg-blue-600 hover:bg-blue-700">
          Enviar nova mensagem
        </Button>
      </div>
    )
  }

  return (
    <form onSubmit={handleSubmit} className="space-y-4">
      <div>
        <Input
          type="text"
          name="name"
          placeholder="Nome completo"
          value={formState.name}
          onChange={handleChange}
          className={errors.name ? "border-red-500" : ""}
        />
        {errors.name && <p className="text-red-500 text-sm mt-1">{errors.name}</p>}
      </div>

      <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
        <div>
          <Input
            type="email"
            name="email"
            placeholder="Email"
            value={formState.email}
            onChange={handleChange}
            className={errors.email ? "border-red-500" : ""}
          />
          {errors.email && <p className="text-red-500 text-sm mt-1">{errors.email}</p>}
        </div>
        <div>
          <Input
            type="tel"
            name="phone"
            placeholder="Telefone"
            value={formState.phone}
            onChange={handleChange}
            className={errors.phone ? "border-red-500" : ""}
          />
          {errors.phone && <p className="text-red-500 text-sm mt-1">{errors.phone}</p>}
        </div>
      </div>

      <div>
        <Select value={formState.service} onValueChange={handleSelectChange}>
          <SelectTrigger className={errors.service ? "border-red-500" : ""}>
            <SelectValue placeholder="Selecione o serviço desejado" />
          </SelectTrigger>
          <SelectContent>
            <SelectItem value="corporate">Eventos Corporativos</SelectItem>
            <SelectItem value="tourism">Turismo & Excursões</SelectItem>
            <SelectItem value="school">Eventos Escolares</SelectItem>
            <SelectItem value="religious">Eventos Religiosos</SelectItem>
            <SelectItem value="employees">Transporte de Funcionários</SelectItem>
            <SelectItem value="sports">Eventos Esportivos</SelectItem>
          </SelectContent>
        </Select>
        {errors.service && <p className="text-red-500 text-sm mt-1">{errors.service}</p>}
      </div>

      <div>
        <Textarea
          name="message"
          placeholder="Descreva sua necessidade de transporte..."
          value={formState.message}
          onChange={handleChange}
          rows={4}
        />
      </div>

      <Button type="submit" className="w-full bg-blue-600 hover:bg-blue-700" disabled={isSubmitting}>
        {isSubmitting ? "Enviando..." : "Enviar mensagem"}
      </Button>
    </form>
  )
}

